#include <iostream>
#include <iomanip>
#include <cmath>
#include <algorithm>
#include <set>
#include <fstream>
#include <sstream>
#include <string>

class Picture
{
private:
  int **matrix;
  int width;
  int height;
  std::string name;

public:
  /**
   * Default constructor. Initializes a Picture object with
   * default values for width, height, and name. The matrix is
   * set to nullptr, and the name is set to "Unnamed".
   */
  Picture() : matrix(nullptr), width(0), height(0), name("Unnamed") {}

  /**
   * Constructs a Picture object with the specified width, height, and optional name.
   * Initializes a matrix of size height x width, with each element set to 0.
   * If the width or height is non-positive, an exception is thrown.
   *
   * @param w Width of the picture.
   * @param h Height of the picture.
   * @param n Name of the picture, default is "Unnamed".
   * @throws std::invalid_argument if width or height is non-positive.
   */
  Picture(int w, int h, const std::string &n = "Unnamed") : width(w), height(h), name(n)
  {
    if (w <= 0 || h <= 0)
    {
      throw std::invalid_argument("Dimenzije slike moraju biti pozitivne.");
    }
    matrix = new int *[height];
    for (int i = 0; i < height; ++i)
    {
      matrix[i] = new int[width]{0};
    }
  }

  /**
   * Destructor - Frees the memory allocated by the Picture object.
   */
  ~Picture()
  {
    for (int i = 0; i < height; ++i)
    {
      delete[] matrix[i];
    }
    delete[] matrix;
  }

  /**
   * Copy constructor - Copies the contents of another Picture object into this one.
   * The name and dimensions of the other picture are copied, and the matrix is
   * copied element by element.
   *
   * @param other Picture object which is copied
   */
  Picture(const Picture &other) : width(other.width), height(other.height), name(other.name)
  {
    matrix = new int *[height];
    for (int i = 0; i < height; ++i)
    {
      matrix[i] = new int[width];
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = other.matrix[i][j];
      }
    }
  }

  /**
   * Returns the dimensions of the picture as a pair of integers.
   * The first element is the width and the second element is the height.
   *
   * @return A pair containing the width and height of the picture.
   */
  inline std::pair<int, int> getDimensions() const
  {
    return {width, height};
  }

  /**
   * Increases the brightness of the picture by the given amount. The amount
   * to increase the brightness by is given as a parameter. The picture elements
   * are increased by this amount, and the result is clamped to the range 0-512.
   *
   * @param s Amount to increase the brightness by
   */
  void Brightness(int s)
  {
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = std::clamp(matrix[i][j] + s, 0, 512);
      }
    }
  }

  /**
   * Reads the picture elements from the standard input. The picture must be
   * entered as a sequence of three-digit numbers, each number representing
   * one element of the picture. The numbers are read row by row, and each row
   * is read from left to right. The input is validated to ensure that all
   * elements are in the range 0-512.
   */
  void inputPicture(bool withName = false, bool restricted = true)
  {
    if (withName)
    {
      std::cout << "Unesite ime slike: ";
      std::cin >> name;
    }
    std::cout << "Unesite piksele slike (" << width << "x" << height << "):\n";
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        std::cin >> matrix[i][j];
        if (restricted && !(matrix[i][j] >= 0 && matrix[i][j] <= 1))
        {
          throw std::invalid_argument("Pikseli slike moraju biti 0 ili 1.");
        }
      }
    }
  }

  /**
   * Reads the picture elements from a specified text file. The picture must be
   * entered as a sequence of three-digit numbers, each number representing
   * one element of the picture. The numbers are read row by row, and each row
   * is read from left to right. The input is validated to ensure that all
   * elements are in the range 0-512.
   *
   * @param filename The name of the text file to read the picture from.
   * @throws std::runtime_error if the file cannot be opened.
   *
   * filename should contain:
   * 0 0 0
   * 0 0 0
   * 0 0 0
   * depends on size and pixel values
   *
   */
  void pictureFromFile(const std::string &filename)
  {
    std::ifstream file(filename);
    if (!file.is_open())
    {
      throw std::runtime_error("Datoteka " + filename + " ne moze biti otvorena.");
    }
    std::string line;
    std::getline(file, line);
    std::istringstream iss(line);
    int w = 0;
    for (std::string token; std::getline(iss, token, ' ');)
    {
      ++w;
    }
    if (w == 0)
    {
      throw std::runtime_error("Datoteka " + filename + " nema piksela.");
    }
    width = w;
    height = 0;
    while (std::getline(file, line))
    {
      if (height == 0)
        ++height;
      ++height;
    }
    if (height == 0)
    {
      throw std::runtime_error("Datoteka " + filename + " nema piksela.");
    }
    file.clear();
    file.seekg(0);
    matrix = new int *[height];
    for (int i = 0; i < height; ++i)
    {
      matrix[i] = new int[width];
      for (int j = 0; j < width; ++j)
      {
        file >> matrix[i][j];
      }
    }
    file.close();
  }

  /**
   * Overlays another picture onto this one. The size of the two pictures
   * must be the same for the overlay to be performed. Each element of this
   * picture's matrix is combined with the corresponding element of the other
   * picture's matrix by taking their average.
   *
   * @param other The Picture object to overlay onto this one.
   * @throws std::invalid_argument if the dimensions of the pictures do not match.
   */
  void overlay(const Picture &other)
  {
    if (width != other.width || height != other.height)
    {
      throw std::invalid_argument("Dimenzije slika moraju biti iste za preklapanje.");
    }

    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = (matrix[i][j] + other.matrix[i][j]) / 2;
      }
    }
  }

  /**
   * Counts the number of unique colors in the picture.
   * Iterates over each element in the picture matrix and adds it
   * to a set to determine the number of distinct colors.
   *
   * @return The number of unique colors present in the picture matrix.
   */
  int countUniqueColors() const
  {
    std::set<int> uniqueColors;
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        uniqueColors.insert(matrix[i][j]);
      }
    }
    return uniqueColors.size();
  }

  /**
   * Assigns the contents of another Picture object to this one.
   * After this call, the contents of the two Picture objects are equal.
   *
   * @param other The Picture object to copy from.
   * @return A reference to the current object.
   */
  Picture &operator=(const Picture &other)
  {
    if (this == &other)
      return *this;
    for (int i = 0; i < height; ++i)
      delete[] matrix[i];
    delete[] matrix;

    width = other.width;
    height = other.height;
    name = other.name;

    matrix = new int *[height];
    for (int i = 0; i < height; ++i)
    {
      matrix[i] = new int[width];
      std::copy(other.matrix[i], other.matrix[i] + width, matrix[i]);
    }
    return *this;
  }

  /**
   * Copies the contents of the right-hand side Picture object to the left-hand side Picture object.
   *
   * @param other The Picture object to copy from.
   * @return A reference to the current object.
   */
  friend std::istream &operator>>(std::istream &in, Picture &pic)
  {
    for (int i = 0; i < pic.height; ++i)
    {
      for (int j = 0; j < pic.width; ++j)
      {
        in >> pic.matrix[i][j];
      }
    }
    return in;
  }

  /**
   * Overloads the extraction operator to read the elements of a Picture object
   * from an input stream. The elements are read row by row, from left to right.
   *
   * @param in  The input stream to read from.
   * @param pic The Picture object to store the read elements.
   * @return The input stream.
   */
  friend std::ostream &operator<<(std::ostream &out, const Picture &pic)
  {
    out << "Picture Name: " << pic.name << "\n";
    for (int i = 0; i < pic.height; ++i)
    {
      for (int j = 0; j < pic.width; ++j)
      {
        out << pic.matrix[i][j] << " ";
      }
      out << "\n";
    }
    return out;
  }

  /**
   * Prints the picture to the standard output. Each element of the picture
   * is printed as a four-digit number, and each row is printed on a separate
   * line.
   */
  void displayPicture() const
  {
    std::cout << "Picture Name: " << name << "\n";
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        std::cout << std::setw(4) << matrix[i][j] << " ";
      }
      std::cout << "\n";
    }
  }

  /**
   * Resizes the picture to the given dimensions. The new picture is created
   * by interpolating the old one using the nearest neighbor method.
   *
   * @param nWidth  New width of the picture
   * @param nHeight New height of the picture
   */
  void resize(int nWidth, int nHeight)
  {
    double nXFactor = static_cast<double>(width) / nWidth;
    double nYFactor = static_cast<double>(height) / nHeight;

    int **newMatrix = new int *[nHeight];
    for (int i = 0; i < nHeight; ++i)
    {
      newMatrix[i] = new int[nWidth];
      for (int j = 0; j < nWidth; ++j)
      {
        int srcX = static_cast<int>(std::floor(j * nXFactor));
        int srcY = static_cast<int>(std::floor(i * nYFactor));
        newMatrix[i][j] = matrix[srcY][srcX];
      }
    }

    for (int i = 0; i < height; ++i)
    {
      delete[] matrix[i];
    }
    delete[] matrix;

    matrix = newMatrix;
    width = nWidth;
    height = nHeight;
  }

  /**
   * Inverts the picture, i.e. all elements are subtracted from 512.
   */
  void invert()
  {
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = 512 - matrix[i][j];
      }
    }
  }

  /**
   * Sets the name of the picture.
   * @param n The new name of the picture.
   */
  void setName(const std::string &n) { name = n; }

  /**
   * Sets the name of the picture by prompting the user for input.
   * The input is read from standard input and stored in the name field.
   */
  void inputName()
  {
    std::cout << "Enter the name of the picture: ";
    std::cin >> name;
  }
};

class Collection
{
private:
  int maxPictures;
  int currentCount;
  Picture *pictures;

public:
  /**
   * Constructor - Initializes a new Collection object with a specified maximum number of pictures.
   *
   * @param max The maximum number of pictures that can be stored in the collection.
   */
  Collection(int max) : maxPictures(max), currentCount(0)
  {
    pictures = new Picture[maxPictures];
  }

  /**
   * Destructor - Frees the memory allocated by the Collection object.
   */
  ~Collection()
  {
    delete[] pictures;
  }

  /**
   * Adds a new picture to the collection. Throws an overflow_error
   * if the collection is full.
   *
   * @param pic The picture to add to the collection
   */
  void addPicture(const Picture &pic)
  {
    if (currentCount >= maxPictures)
    {
      throw std::overflow_error("Kolekcija je puna.");
    }
    pictures[currentCount++] = pic;
  }

  /**
   * Adds a new picture to the collection. Throws an overflow_error
   * if the collection is already full. The picture is added to the
   * next available position in the collection.
   *
   * @param pic The Picture object to be added to the collection.
   * @throws std::overflow_error if the collection is full.
   */
  friend std::ostream &operator<<(std::ostream &out, const Collection &col)
  {
    for (int i = 0; i < col.currentCount; ++i)
    {
      out << "--------------------------------\n"
          << "Picture " << i + 1 << ":\n"
          << col.pictures[i];
    }
    out << "--------------------------------\n";
    return out;
  }
};

int main()
{
  Collection imageCollection(8);

  // Create Picture object size 10x10
  Picture originalImage(3, 3, "Originalna");
  originalImage.inputPicture(false, false);
  imageCollection.addPicture(originalImage);

  // Invert
  Picture invertedImage(originalImage);
  invertedImage.invert();
  invertedImage.setName("Obrnuta");
  imageCollection.addPicture(invertedImage);

  // Decrease Brightness by 20
  Picture dimmedImage(originalImage);
  dimmedImage.Brightness(-20);
  dimmedImage.setName("Zatamnjena");
  imageCollection.addPicture(dimmedImage);

  // Copy the picture to another
  Picture duplicateImage = originalImage;
  duplicateImage.setName("Duplirana");
  imageCollection.addPicture(duplicateImage);

  // Resize the copied picture to 15x15
  Picture resizedImage(duplicateImage);
  resizedImage.resize(5, 5);
  resizedImage.setName("Povecana");
  imageCollection.addPicture(resizedImage);

  // Use operators for pictures for last image
  Picture operatorImage = originalImage;
  operatorImage.setName("Ucitana");
  std::cout << "Ucitajte drugu sliku:\n";
  std::cin >> operatorImage;
  imageCollection.addPicture(operatorImage);

  // Overlay the first with the last image
  Picture overlaidImage = operatorImage;
  overlaidImage.overlay(originalImage);
  overlaidImage.setName("Preklapana (Original + Ucitana)");
  imageCollection.addPicture(overlaidImage);

  // Load a picture from picture.txt
  Picture loadedImage;
  loadedImage.pictureFromFile("picture.txt");
  loadedImage.setName("Ucitana iz datoteke");
  imageCollection.addPicture(loadedImage);

  std::cout << "Kolekcija:\n"
            << imageCollection;

  return 0;
}
